//*************************************************************************************************
//
//	Description:
//		basic_instanced_billboard.fx - Instanced and billboarded version of basic.fx
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     31/07/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#if !defined( _3DSMAX_ )
#define	INSTANCING
#endif
#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#if !defined( _3DSMAX_ )
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	bool appEdit = false;
>;
#endif


//
// Platform specific data for instancing
//
#if defined( INSTANCING ) && defined( _XBOX )
float4 instance_data		// x=no of indices per instance, y=start of index data within IB
<
	bool appEdit = false;
>;
#endif

#if defined(INSTANCING) && defined(_PS3_)
float4x4 instanceXForm
<
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( INSTANCING )

SHARE_PARAM float4x4 view : View
<
	bool appEdit = false;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
>;

SHARE_PARAM float4x4 viewI : ViewI
<
	bool appEdit = false;
>;

#else

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 viewI : ViewI
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 projMatrix : Projection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

#endif



SPECIALISATION_PARAM_DEFAULTS_TRUE( useZLock, "Use locked Z?", "LOCK_Z" )	// TRUE if specialisation for up axis locking should be used
SPECIALISATION_PARAM( useYLock, "Use locked Y?", "LOCK_Y" )	// TRUE if specialisation for up axis locking should be used
SPECIALISATION_PARAM( useXLock, "Use locked X?", "LOCK_X" )	// TRUE if specialisation for up axis locking should be used
SPECIALISATION_PARAM( useArbitraryLock, "Use (LockX, LockY, LockZ)?", "LOCK_ARB" )	// TRUE if arbitrary lock should be used

//
// Using 3 floats is far from ideal, but specialisations do not seem to work for this case (compiler bug, too many ifs) and max does not like float3
//
#if defined( _3DSMAX_ ) || defined( LOCK_ARB )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( lockX, "lockX", useArbitraryLock, -1.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( lockY, "lockY", useArbitraryLock, -1.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( lockZ, "lockZ", useArbitraryLock, -1.0f, 1.0f, 0.0f )
#endif

//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_
// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Second UV holds the billboarding pivot offsets
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
	bool use3dMapping = true;
> = 0;

// Vertex colour channel
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

SPECIALISATION_PARAM( useAnisotropic, "Anisotropic Filter?", "USE_ANISO" )	// TRUE if the anisotropic filtering is to be used

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS


//
// This shader supports variance 3 mask tint
//
#include "variance_3mask_tint.fxh"

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

#if defined( USE_ANISO )

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 6;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 6 )
#endif
};

#else

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
	float3 billboardPivot: TEXCOORD1;
	
	float3 colour   : TEXCOORD2;													// Vertex colour																											// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD3;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float3 billboardPivot: TEXCOORD1;
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// Texture coords
	float3 billboardPivot: TEXCOORD1;

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;												// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1;								// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 4 )
#endif
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 coords				: TEXCOORD1;
};

struct ZPRIME_VSOUTPUT
{
	float4 position			: POSITION;
};

//-----------------------------------------------------------------------
//
// Vertex shader code
//

float3 LockBillboardingAxis( float3 _result, float3 _billboardLockAxis, float3 _billboardPivot, float4x4 _view )
{
#ifdef _3DSMAX_
	float3 upAxis = mul( _billboardLockAxis, (float3x3)_view );
	float3 rightAxis = normalize(cross( upAxis, _result ));
	float3 forwardAxis = cross( rightAxis, upAxis );	
	return ( _billboardPivot.x * rightAxis + _billboardPivot.y * upAxis + _billboardPivot.z * forwardAxis );
	
#else
	float3 upAxis = mul( _billboardLockAxis, (float3x3)_view );	
	float3 rightAxis = normalize(cross( upAxis, _result ));
	float3 forwardAxis = cross( rightAxis, upAxis );	

	return ( _billboardPivot.x * rightAxis + _billboardPivot.y * upAxis + _billboardPivot.z * forwardAxis );
#endif	
	

	return _result;
}

float3 DoBilboarding( float3 _inputPosition, float3 _billboardPivot, float4x4 _worldview, float4x4 _view )
{
	#ifdef _3DSMAX_
		_billboardPivot.y += 1;
		float3 pivotPoint = _inputPosition - _billboardPivot;
	#else
		float3 pivotPoint = _inputPosition - _billboardPivot.xzy;
	#endif

	float3 result = mul( float4( pivotPoint, 1.0f ), _worldview ).xyz;
	
	DEPENDENT_CODE_START( useZLock )
	#if defined( _3DSMAX_ ) || defined( LOCK_Z )
	
		#ifdef _3DSMAX_
			result += LockBillboardingAxis( result, float3( 0, 0, 1 ), _billboardPivot, _view );
		#else
			result += LockBillboardingAxis( result, float3( 0, 1, 0 ), _billboardPivot, _view );	
		#endif

	#endif
	DEPENDENT_CODE_ELSE( useZLock )
	#if defined( _3DSMAX_ ) || !defined( LOCK_Z )

		DEPENDENT_CODE_START( useYLock )
		#if defined( _3DSMAX_ ) || defined( LOCK_Y )
		
			#ifdef _3DSMAX_
				result += LockBillboardingAxis( result, float3( 0, 1, 0 ), _billboardPivot, _view );
			#else
				result += LockBillboardingAxis( result, float3( 0, 0, 1 ), _billboardPivot, _view );	
			#endif
		
		#endif
		DEPENDENT_CODE_ELSE( useYLock )
		#if defined( _3DSMAX_ ) || !defined( LOCK_Y )

			DEPENDENT_CODE_START( useXLock )
			#if defined( _3DSMAX_ ) || defined( LOCK_X )
				#ifdef _3DSMAX_
					result += LockBillboardingAxis( result, float3( 1, 0, 0 ), _billboardPivot, _view );
				#else
					result += LockBillboardingAxis( result, float3( 1, 0, 0 ), _billboardPivot, _view );	
				#endif
				
			#endif
			DEPENDENT_CODE_ELSE( useXLock )
			#if defined( _3DSMAX_ ) || !defined( LOCK_X )

							DEPENDENT_CODE_START( useArbitraryLock )
							#if defined( _3DSMAX_ ) || defined( LOCK_ARB )
							
								#ifdef _3DSMAX_
									result += LockBillboardingAxis( result, float3( lockX, lockY, lockZ ), _billboardPivot, _view );
								#else
									result += LockBillboardingAxis( result, float3( lockX, lockZ, lockY ), _billboardPivot, _view );	
								#endif
								
							#endif
							DEPENDENT_CODE_ELSE( useArbitraryLock )
							#if defined( _3DSMAX_ ) || !defined( LOCK_ARB )
							
								_billboardPivot.z = -_billboardPivot.z;
								result += _billboardPivot.xyz;
								
							#endif
							DEPENDENT_CODE_END( useArbitraryLock )
							
			#endif
			DEPENDENT_CODE_END( useXLock )
									
		#endif
		DEPENDENT_CODE_END( useYLock )
							
	#endif
	DEPENDENT_CODE_END( useZLock )

	return result;
}

#ifdef _POST_PARSE_
VSOUTPUT BasicVertexShader( VSINPUT _input )
{
#else
#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT BasicVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fbillboardPivot;
	float4 fnormal;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fbillboardPivot, iMeshIndex, texcoord1;
        vfetch fnormal, iMeshIndex, normal0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.billboardPivot = fbillboardPivot.xyz;
	_input.normal = fnormal.xyz;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = ftangent.xyz;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
VSOUTPUT BasicVertexShader( VSINPUT _input )
{
#endif
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

	float4x4	worldview = mul( world, view );

	VSOUTPUT _output;

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// billboarding	
	float3 viewPos = DoBilboarding( _input.position, _input.billboardPivot.xyz, worldview, view );

	// Calculate vert's world position
	float3 worldPos = mul( float4( viewPos, 1.0f ), viewI ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_and_fresnel = float4( normal, CalculateFresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_and_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#else
	DEPENDENT_CODE_START( useSpecular )
	float globalSpecularFactorValue = globalSpecularFactor;
	DEPENDENT_CODE_END( useSpecular )
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
SHADOWGEN_VSOUTPUT BasicInstancedShadowGenVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	SHADOWGEN_VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fbillboardPivot;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fbillboardPivot, iMeshIndex, texcoord1;
        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.billboardPivot = fbillboardPivot.xyz;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
SHADOWGEN_VSOUTPUT BasicInstancedShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

	float4x4	worldview = mul( world, view );

	SHADOWGEN_VSOUTPUT _output;

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;
	
	float3 viewPos = DoBilboarding( _input.position, _input.billboardPivot.xyz, worldview, view );
		
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}


#if defined(INSTANCING) && defined(_XBOX)
ZPRIMEDOF_VSOUTPUT BasicInstancedZPrimeDOFVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	SHADOWGEN_VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fbillboardPivot;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fbillboardPivot, iMeshIndex, texcoord1;
        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.billboardPivot = fbillboardPivot.xyz;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
ZPRIMEDOF_VSOUTPUT BasicInstancedZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

	float4x4	worldview = mul( world, view );

	ZPRIMEDOF_VSOUTPUT _output;

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	float3 viewPos = DoBilboarding( _input.position, _input.billboardPivot.xyz, worldview, view );
	
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );
	
	
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD3;						// Vertex colour
	float2 texCoord		: TEXCOORD0;						// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1;		// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;						// Eye vector (world space)
	float3 tangent		: TEXCOORD4;						// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;						// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD3;									// Vertex colour
	float2 texCoord		: TEXCOORD0;									// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1_centroid;	// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2_centroid;					// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4_centroid;					// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 4 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = Tint_DoIfEnabled( tex2D( diffuseMap, _input.texCoord ), _input.texCoord, false );
	float4 specularTexColour;

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord );

		DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
			globalSpecularFactorValue = 0.0f;
#endif
		DEPENDENT_CODE_END( useEnvironmentMap )

#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to ones (will be optimised out)
		specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_and_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	//
	// Calculate the fresnel factor
	//
	float fresnelCoefficient;
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		fresnelCoefficient = saturate( _input.normal_and_fresnel.w );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		fresnelCoefficient = 1.0f;
#endif
	DEPENDENT_CODE_END( useFresnel )

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		float3 reflectionVector = reflect( -eye, normal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor * fresnelCoefficient;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: COLOR0;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


#ifdef _POST_PARSE_
VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
#else
#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT_LD BasicLowDetailVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fbillboardPivot;
	float4 fnormal;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fbillboardPivot, iMeshIndex, texcoord1;
        vfetch fnormal, iMeshIndex, normal0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.billboardPivot = fbillboardPivot.xyz;
	_input.normal = fnormal.xyz;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = ftangent.xyz;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
#endif
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

	float4x4	worldview = mul( world, view );

	VSOUTPUT_LD _output;

	_output.texCoord = _input.texCoord;

	// billboarding	
	float3 viewPos = DoBilboarding( _input.position, _input.billboardPivot.xyz, worldview, view );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( viewPos, 1.0f ), projMatrix );

	// Get normal and position in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	float3 worldPos = mul( float4( viewPos, 1.0f ), viewI ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = Tint_DoIfEnabled( tex2D( diffuseMap, _input.texCoord ), _input.texCoord, true );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicInstancedShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = diffuseTexColour.a * _input.colour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicInstancedZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a * _input.colour.a;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicZPrimeFragmentShader() : COLOR0
{
	PSOUTPUT output;
	output.Colour = 1;
	return output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Basic
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool isBillboard = true;
	bool instancedShader = true;
	bool disableAmbientShadow = true;
	bool   appCanOverrideSampler = true;
	
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic";
	int    normalDeferredID		= 0;

	string zprimeBehaviour		= "ERMB_DONT_RENDER";

	string zprimeDOFBehaviour	= "ERMB_RENDER";
	string zprimeDOFTechnique	= "_Basic_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;

	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique	= "_Basic_ShadowGen";
	int    shadowGenDeferredID	= 0;
	
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Basic_LowDetail";
	int    lowDetailDeferredID = 0;
	
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}


technique Basic_AlphaTest
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool isBillboard = true;
	bool instancedShader = true;
	bool disableAmbientShadow = true;
	bool   appCanOverrideSampler = true;
	
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_Basic_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Basic_LowDetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = true;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif


#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique Basic_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool isBillboard = true;
	bool instancedShader = true;
	bool disableAmbientShadow = true;

	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx BasicLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicLowDetailVertexShader();
		PixelShader = compile ps_3_0 BasicLowDetailFragmentShader();
#endif
	}
}


technique _Basic_ShadowGen
<
	bool isBillboard = true;
	bool instancedShader = true;
>
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 64 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicInstancedShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedShadowGenFragmentShader();
#elif defined (_XBOX)
			VertexShader = compile vs_3_0 BasicInstancedShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicInstancedShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedShadowGenFragmentShader();
#endif
	}
}


technique _Basic_ZPrime_DOF
<
	bool preservesGlobalState = true;
	bool isBillboard = true;
	bool instancedShader = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = false;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicInstancedZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = false;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicInstancedZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedZPrimeDOFFragmentShader();
#endif
	}
}
